<?php
ob_start();
session_start();
require_once "config/db.php";

if (!isset($_SESSION['admin'])) {
    header("Location: login.php");
    exit;
}

$success_message = '';
$error_message = '';

$category_name = 'MPCB';
$subcategory_name = 'Bank Guarantee Validity';

$freq_months = [
    "monthly"      => 1,
    "quarterly"    => 3,
    "six_monthly"  => 6,
    "yearly"       => 12
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $company_id    = isset($_POST['company_id']) ? (int)$_POST['company_id'] : 0;
    $reminder_date = $_POST['reminder_date'] ?? '';
    $note          = trim($_POST['note'] ?? '');
    $frequencies   = $_POST['frequency'] ?? [];

    if ($company_id <= 0) {
        $error_message = "Please select a valid company.";
    } elseif (empty($reminder_date)) {
        $error_message = "Please select a valid date.";
    } elseif (empty($frequencies)) {
        $error_message = "Please select at least one reminder frequency.";
    } else {
        $insert_sql = "INSERT INTO company_reminders 
            (company_id, category_name, subcategory_name, reminder_date, expiry_date, note, frequency_type, status, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', NOW())";
        $stmt = $conn->prepare($insert_sql);
        if (!$stmt) {
            $error_message = "Prepare failed: " . $conn->error;
        } else {
            $hadError = false;
            foreach ($frequencies as $freq) {
                if (!isset($freq_months[$freq])) continue;
                $months = $freq_months[$freq];
                $expiry_date = date('Y-m-d', strtotime("+$months months", strtotime($reminder_date)));
                $stmt->bind_param(
                    "issssss",
                    $company_id,
                    $category_name,
                    $subcategory_name,
                    $reminder_date,
                    $expiry_date,
                    $note,
                    $freq
                );
                if (!$stmt->execute()) {
                    $hadError = true;
                    error_log("Insert failed for freq {$freq}: " . $stmt->error);
                }
            }
            $stmt->close();
            if ($hadError) {
                $error_message = "Some reminders could not be saved. Check error log.";
            } else {
                $_SESSION['success_message'] = "Bank Guarantee reminder(s) added successfully!";
                header("Location: " . $_SERVER['PHP_SELF']);
                exit;
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Add Bank Guarantee Validity - Axle_Services</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet" />
<link rel="stylesheet" href="css/style.css" />
<link rel="stylesheet" href="css/header.css" />
<link rel="stylesheet" href="css/sidebar.css" />
<style>
body { background: #f9fafb; color: #333; min-height: 100vh; }
.main-content {
    margin-top: 70px; margin-left: 250px; padding: 28px 34px;
    background: #fff; border-radius: 12px; box-shadow: 0 8px 24px rgba(0,0,0,0.05);
}
@media (max-width: 991.98px) { .main-content { margin-left:0; padding:20px; border-radius:0; box-shadow:none; } }
.page-header {
    background-color: #0d6efd; color: #fff; padding: 18px 22px;
    border-radius: 12px; margin-bottom: 24px; text-align: center;
    font-weight: 600; font-size: 1.5rem;
}
form { max-width: 900px; margin:0 auto; }
label.form-label { font-weight: 600; color: #495057; }
input.form-control, select.form-select, textarea.form-control {
    border-radius:8px; border:1.5px solid #ced4da; padding:10px 14px; font-size:1rem;
}
input.form-control:focus, textarea.form-control:focus, select.form-select:focus {
    border-color:#0d6efd; box-shadow:0 0 8px rgba(13,110,253,0.18); outline:none;
}
.btn-success { background-color:#198754; border-color:#198754; font-weight:600; font-size:1.05rem; padding:10px 36px; border-radius:999px; }
.btn-success:hover { background-color:#157347; }
.alert { max-width:900px; margin:0 auto 20px; border-radius:8px; font-weight:600; }
#companyList { z-index:2000; width:100%; max-height:320px; overflow:auto; }

/* Frequency cards */
.freq-card { display:inline-block; padding:12px 14px; border-radius:10px; border:1px solid #e3e7ee; margin-right:10px; margin-bottom:8px; min-width:160px; vertical-align:top; background:#fafbfc; }
.freq-card input { margin-top:2px; }
.freq-card small { color:#6c757d; }
@media (max-width:575.98px) { .freq-card { min-width:48%; display:block; } }
</style>
</head>
<body>
<?php include 'header.php'; ?>
<?php include 'sidebar.php'; ?>

<div class="main-content">
  <div class="page-header">Add Bank Guarantee Validity</div>

  <?php if (!empty($error_message)): ?>
    <div class="alert alert-danger"><?= htmlspecialchars($error_message) ?></div>
  <?php endif; ?>
  <?php if (!empty($_SESSION['success_message'])): ?>
    <div class="alert alert-success"><?= htmlspecialchars($_SESSION['success_message']) ?></div>
    <?php unset($_SESSION['success_message']); ?>
  <?php endif; ?>

  <form method="POST" class="row g-4 needs-validation" novalidate>
    <!-- Company Search -->
    <div class="row align-items-center mb-3 mt-3">
      <div class="col-md-4">
        <label class="form-label">Select Company</label>
      </div>
      <div class="col-md-8 position-relative">
        <input type="text" id="companySearch" class="form-control" placeholder="Type company name..." autocomplete="off" required>
        <input type="hidden" name="company_id" id="companyId">
        <div id="companyList" class="list-group position-absolute bg-light shadow-sm" style="display:none;"></div>
      </div>
    </div>

    <!-- Reminder Date -->
    <div class="row">
      <div class="col-md-4">
        <label class="form-label">Renewal Date</label>
      </div>
      <div class="col-md-4">
        <input type="date" name="reminder_date" class="form-control" value="<?= date('Y-m-d'); ?>" required>
      </div>
    </div>

    <!-- Frequency -->
    <div class="row">
      <div class="col-md-4">
        <label class="form-label">Reminder Frequency</label>
      </div>
      <div class="col-md-8">
        <label class="freq-card">
          <input class="form-check-input me-2" type="checkbox" name="frequency[]" value="monthly">
          <strong>Monthly</strong><br><small>Every 1 month</small>
        </label>
        <label class="freq-card">
          <input class="form-check-input me-2" type="checkbox" name="frequency[]" value="quarterly">
          <strong>Quarterly</strong><br><small>Every 3 months</small>
        </label>
        <label class="freq-card">
          <input class="form-check-input me-2" type="checkbox" name="frequency[]" value="six_monthly">
          <strong>Six Monthly</strong><br><small>Every 6 months</small>
        </label>
        <label class="freq-card">
          <input class="form-check-input me-2" type="checkbox" name="frequency[]" value="yearly">
          <strong>Yearly</strong><br><small>Every 12 months</small>
        </label>
      </div>
    </div>

    <!-- Note -->
    <div class="row">
      <div class="col-md-4">
        <label class="form-label">Note (Optional)</label>
      </div>
      <div class="col-md-8">
        <textarea name="note" class="form-control" rows="3" placeholder="Enter any notes or remarks..."><?= htmlspecialchars($_POST['note'] ?? '') ?></textarea>
      </div>
    </div>

    <div class="col-12 text-center mt-3">
      <button type="submit" class="btn btn-success">Save</button>
    </div>
  </form>
</div>

<script>
// company search autocomplete
const companySearch = document.getElementById("companySearch");
const companyIdInput = document.getElementById("companyId");
const companyList = document.getElementById("companyList");

let abortController = null;
companySearch.addEventListener("input", function() {
    const q = this.value.trim();
    if (abortController) abortController.abort();
    if (q.length < 1) { companyList.style.display="none"; companyList.innerHTML=""; companyIdInput.value=""; return; }
    abortController = new AbortController();
    fetch("get_companies.php?query=" + encodeURIComponent(q), { signal: abortController.signal })
        .then(r => r.json())
        .then(data => {
            if (!Array.isArray(data) || data.length === 0) { companyList.innerHTML="<div class='list-group-item'>No companies found</div>"; companyList.style.display="block"; return; }
            companyList.innerHTML = data.map(c=> `<button type="button" class="list-group-item list-group-item-action" data-id="${c.id}" data-name="${c.name}">${c.name}</button>`).join('');
            companyList.style.display="block";
            document.querySelectorAll("#companyList .list-group-item-action").forEach(btn=>{
                btn.addEventListener("click", function() {
                    companySearch.value=this.dataset.name;
                    companyIdInput.value=this.dataset.id;
                    companyList.innerHTML="";
                    companyList.style.display="none";
                });
            });
        }).catch(err=>{if(err.name!=='AbortError') console.error(err);});
});
document.addEventListener("click", function(e){ if(!companyList.contains(e.target) && e.target!==companySearch){ companyList.style.display="none"; companyList.innerHTML=""; }});
</script>

<script src="/js/script.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
<?php ob_end_flush(); ?>
